const chalk = require("chalk");
const ZShepherd = require("zs-gateway");
const deviceDAO = require("../../../database/models/DAO/device.dao");
const environmentDAO = require("../../../database/models/DAO/environment.dao");
const ruleDAO = require("../../../database/models/DAO/rule.dao");
const scenarioDAO = require("../../../database/models/DAO/scenario.dao");
const msgDispatcher = require("../shepherd/zs-messageDispatcher");
var gpioControl = require("../../../system-on-module/gpioController");
const deviceProcessor = require("../services/processor.device");
const util = require("util");
const delay = require('delay');
const firmware = require("../services/processor.firmware");
const shell = require('shelljs');
var _ = require("lodash");
const processorStatus = require('../services/processor.status')

let joining = false;

module.exports.refresh = shepRefresh;
module.exports.reset = resetShep;
module.exports.join = joinShep;
module.exports.functional = functional;
module.exports.bind = bind;
module.exports.unbind = unbind;
module.exports.write = write;
module.exports.shepRead = shepRead;
module.exports.read = read;
module.exports.remove = remove;
module.exports.removeFromShep = removeFromShep;
module.exports.list = list;
module.exports.updateEnvironments = updateEnvironments;
module.exports.updateScenarios = updateScenarios;
module.exports.updateRules = updateRules;

const shepherd = new ZShepherd("/dev/ttyS1", {
  net: {
    panId: 0xFFFF
  },
  dbPath: `${shell.env.HOME}/ecomfort-db/collections/dev.db`
});

gpioControl.button3.watch(function (err, value) {
  console.log("### RADIO2 button - Permit Join:");
  gpioControl.setLed(3, value, 1, 0);
  if (err) {
    console.log("RADIO2 button ERROR", err);
  } else {
    if (value == 1) {
      console.log("RADIO2 button: ", value);
      //joinShep(90);
    } else if (value == 0) {
      console.log("RADIO2 button: ", value);
    }
  }
});

gpioControl.button2.watch(function (err, value) {
  console.log("### RADIO1 button - Permit Join:");
  if (err) {
    console.log("RADIO1 button ERROR: ", err);
  } else {
    if (value == 1) {
      console.log("RADIO1 button: ", value);
      joinShep(90);
    } else if (value == 0) {
      console.log("RADIO1 button: ", value);
    }
  }
});
                                 
gpioControl.button1.watch(function (err, value) {
  console.log("### REBOOT button:");
  if (err) {        
    console.log("REBOOT button ERROR: ", err);
  } else {                           
    var startTimer;              
    if (value == 1) {          
      console.log("REBOOT button: ", value);
      startTimer = setTimeout(async function () {
        if (gpioControl.button1.readSync() == 1) {
          firmware.rebootGateway();
        }                          
      }, 5000);                   
    } else if (value == 0) {   
      clearTimeout(startTimer);                
      console.log("REBOOT button: ", value);
    }                                                  
  }                                              
}); 

console.log(chalk.blue("[z-shepherd] initializing..."));

shepherd.start(function (err) {
  if (err) {
    console.log(chalk.red("[z-shepherd] [start] shepherd: ", err));
  } else {
    console.log(chalk.green("[z-shepherd] [start] shepherd started"));
    gpioControl.setLed(1, 1, 1, 0);
  }
});

shepherd.on("ready", function () {
  init = true;

  const Zive = require("zive");
  const epInfo = {
    profId: 260,
    devId: 257,
    discCmds: []
  };
  const Ziee = require("ziee");
  let ziee = new Ziee();
  let zive = new Zive(epInfo, ziee);
  shepherd.mount(zive, async function (e, epId) {
    if (e || epId != 8) {
      console.log(
        chalk.red("[z-shepherd] [ready] local endpoint: " + epId + ", should be 8.")
      );

      resetShep(1);
      if (e) console.log(chalk.red("[z-shepherd] [ready] " + e));
    } else {
      console.log(chalk.green("[z-shepherd] [ready] init server is ready."));
      firmware.setImageStatus(1);
      allDevsToOff();
    }
  });
});

shepherd.acceptDevIncoming = function (devInfo, callback) {
  setImmediate(async function () {
      if (joining || await deviceDAO.listDeviceByIeeeAddr({ ieeeAddr: devInfo.ieeeAddr })) {
          console.log(chalk.bgGreen("[z-shepherd] [accept-dev-incoming] " + devInfo.ieeeAddr + ": accepted!"));
          callback(null, true);
      } else {
          console.log(chalk.bgGreen("[z-shepherd] [accept-dev-incoming] " + devInfo.ieeeAddr + ": rejected!"));
          callback(null, false);
      }
  });
};

shepherd.on("ind", async msg => {
  switch (msg.type) {
    case "attReport":
      console.log(
        chalk.blue("[z-shepherd] [attribute-report] ", util.inspect(msg.data))
      );
      try {
        await deviceProcessor.updateAttributes(msg)
      } catch (err) {
        console.log("[z-shepherd] [attribute-report] [error]", err)
	break
      }
      msgDispatcher.messageIncoming(msg, "attReport");
      break;

    case "devChange":
      console.log(chalk.blue("[z-shepherd] [device-change] " + util.inspect(msg.data)));
      msgDispatcher.messageIncoming(msg, "devChange");
      break;

    case "devStatus":
      console.log(chalk.blue("[z-shepherd] [device-status]: " + msg.endpoints[0].getIeeeAddr()));
      msgDispatcher.messageStatus(msg);
      break;

    case "devIncoming":
      console.log(chalk.blue("[z-shepherd] [device-incoming]: ", msg.data));
      let simpleDesc = await msg.endpoints[0].getSimpleDesc();
      let newDevice = {
        ieeeAddr: msg.endpoints[0].device.ieeeAddr,
        epList: msg.endpoints[0].device.epList,
        nwkAddr: msg.endpoints[0].device.nwkAddr,
        clusters: await deviceProcessor.adjustAttributes(
          msg.endpoints[0].clusters
        ),
        devId: simpleDesc.devId.toString(),
        profId: simpleDesc.profId.toString(),
        type: msg.endpoints[0].device.type,
        manufId: msg.endpoints[0].device.manufId.toString(),
        manufName: msg.endpoints[0].device.manufName,
        modelId: msg.endpoints[0].device.modelId,
        powerSource: msg.endpoints[0].device.powerSource,
        status: msg.endpoints[0].device.status,
        inClusterList: simpleDesc.inClusterList,
        outClusterList: simpleDesc.outClusterList
      };
      const device = await deviceDAO.listDeviceByIeeeAddr({
        ieeeAddr: newDevice.ieeeAddr
      });
      if (device) {
        deviceDAO.updateDevice(newDevice, newDevice)
      } else {
        deviceDAO.createDevice(newDevice)
      }
      break;

    case "devLeaving":
      console.log(chalk.bgBlue("[z-shepherd] [device-leaving] " + msg.data));

      updateEnvironments({ ieeeAddr: msg.data });
      updateRules({ ieeeAddr: msg.data }, null);
      updateScenarios({ ieeeAddr: msg.data });
      deviceDAO.deleteDeviceByIeeeAddr({ ieeeAddr: msg.data });
      msgDispatcher.messageLeaving({ ieeeAddr: msg.data });
      break;

    default:
      console.log("[z-shepherd] [incoming-message] [default] ", JSON.stringify(msg));
      gpioControl.setLed(3, 1, 1, 0);
      await delay(200);
      gpioControl.setLed(3, 0, 1, 0);
      await delay(200);
      gpioControl.setLed(3, 1, 1, 0);
      await delay(200);
      gpioControl.setLed(3, 0, 1, 0);
      await delay(200);
      gpioControl.setLed(3, 1, 1, 0);
      await delay(200);
      gpioControl.setLed(3, 0, 1, 0);
      await delay(200);
      break;
  }
});

shepherd.on("permitJoining", function (joinTimeLeft) {
  joining = true;
  if (joinTimeLeft === 0) {
    console.log(chalk.black.bgGreen("[z-shepherd] join disabled"));
    joining = false;
  }
});

async function shepRefresh() {
  return new Promise((resolve, reject) => {
    shepherd.reset(1, function (err) {
      !err
        ? resolve("[z-shepherd] [reset] refresh successfully.")
        : reject("[z-shepherd] [refresh] " + err.toString());
    });
  });
}

async function resetShep(mode) {
  console.log(chalk.blue("[z-shepherd] [reset] reseting..."));

  return new Promise((resolve, reject) => {
    shepherd.reset(Number(mode), function (err) {
      !err
        ? resolve("[z-shepherd] [reset] reset successfully.")
        : reject("[z-shepherd] [reset] " + err.toString());
    });
  });
}

function joinShep(joinTime) {
  gpioControl.setLed(4, 0, 500, joinTime);
  return new Promise((resolve, reject) => {
    shepherd.permitJoin(Number(joinTime), function (err) {
      if (err) {
        console.log(chalk.red("[z-shepherd] [permit-join] join error: ", err));
        reject(err);
      } else {
        console.log(
          chalk.black.bgGreen(
            "[z-shepherd] [permit-join] enabled: " + joinTime + " seconds"
          )
        );
        gpioControl.setLed(4, 0, 1, 0);
        resolve({ time: joinTime });
      }
    });
  });
}

function functional(data) {
  return new Promise((resolve, reject) => {
    shepherd
      .find(data.ieeeAddr, Number(data.epId))
      .functional(data.cId, data.cmd, data.zclData, function (err, callback) {
        err ? reject(err) : resolve(callback);
      });
  });
}

async function bind(input, output) {
  const ep1 = await shepherd.find(input.ieeeAddr, Number(input.epId));
  const ep2 = await shepherd.find(output.ieeeAddr, Number(output.epId));

  const bind = await ep1.bind(input.cId, ep2);

  return bind;
}

async function unbind(input, output) {
  const ep1 = await shepherd.find(input.ieeeAddr, Number(input.epId));
  const ep2 = await shepherd.find(output.ieeeAddr, Number(output.epId));

  const unbind = await ep1.unbind(input.cId, ep2);

  return unbind;
}

function write(data) {
  return new Promise((resolve, reject) => {
    var ep = shepherd.find(data.ieeeAddr, 8);
    ep.write("closuresDoorLock", data.attrId, "" + data.value, function (
      err,
      callback
    ) {
      err ? reject(err) : resolve(callback);
    });
  });
}

function read(data) {
  return new Promise((resolve, reject) => {
    var ep = shepherd.find(data.ieeeAddr, 8);
    ep.read("closuresDoorLock", data.attrId, function (err, callback) {
      err ? reject(err) : resolve(callback);
    });
  });
}

function shepRead(data) {
  return new Promise((resolve, reject) => {
    var ep = shepherd.find(data.ieeeAddr, Number(data.epId));
    ep.read(data.cId, data.attrId, function (err, callback) {
      err ? reject(err) : resolve(callback);
    });
  });
}

function remove(data) {
  return new Promise((resolve, reject) => {
    shepherd.remove(data.ieeeAddr, async function (err, rsp) {
      err ? reject(err.toString()) : resolve(rsp);
    });
  });
}

async function removeFromShep(data) {
  const device = await deviceDAO.listDeviceByIeeeAddr({
    ieeeAddr: data.ieeeAddr
  });
  deviceDAO.deleteDeviceByIeeeAddr(data);
  msgDispatcher.messageLeaving(data);
  const endpoint = await shepherd.find(device.ieeeAddr, device.epList[0]);
  return new Promise((resolve, reject) => {
    shepherd._unregisterDev(endpoint.device, function (err, rsp) {
      err ? reject(err.toString()) : resolve(rsp);
    });
  });

}

async function list(data) {
  return await JSON.stringify(shepherd.list());
}

async function allDevsToOff() {
  const nedbList = await deviceDAO.listDevices();

  for (nedbDevice of nedbList) {
    const device = await deviceDAO.updateDevice(
      { ieeeAddr: nedbDevice.ieeeAddr },
      { status: "offline" }
    );
  }
}

async function updateEnvironments(dev) {
  let device = await deviceDAO.listDeviceByIeeeAddr(dev);
  if (!device) {
    return "O dispositivo não foi encontrado.";
  }
  let environmentA = await environmentDAO.listEnvironmentById(
    device.environment
  );
  let envAfter;
  if (environmentA) {
    const devsA = [];
    for (devA of environmentA.devices) {
      if (devA.ieeeAddr != device.ieeeAddr) devsA.push(devA);
    }
    //console.log(chalk.blue("[z-shepherd] [update-device-env] " + environmentA.name + ": " + JSON.stringify(devsA)));
    envAfter = await environmentDAO.updateEnvironment(
      { environmentId: environmentA.environmentId },
      { devices: devsA }
    );
  }
  return envAfter;
}

async function updateScenarios(dev) {
  let device = await deviceDAO.listDeviceByIeeeAddr(dev);
  if (!device) {
    return "O dispositivo não foi encontrado.";
  }
  let scenarios = await scenarioDAO.listScenarios();
  let scenarioAfter;
  for (scenario of scenarios) {
    let updateResult = await _.remove(scenario.actions, function (o) {
      return o.ieeeAddr == dev.ieeeAddr;
    });
    if (updateResult.length > 0) {
      scenarioAfter = await scenarioDAO.updateScenario(
        { scenarioId: scenario.scenarioId },
        { actions: scenario.actions }
      );
    }
  }

  return scenarioAfter;
}

async function updateRules(dev, scenarioId) {
  let updateInputs;
  let updateOutputs;
  let updateResult;
  if (dev) {
    let device = await deviceDAO.listDeviceByIeeeAddr(dev);
    if (!device) {
      return "O dispositivo não foi encontrado.";
    }
    let rules = await ruleDAO.listRules();
    if (rules) {
      for (rule of rules) {
        for (input of rule.inputs) {
          updateInputs = await _.remove(rule.inputs, function (o) {
            return o.ieeeAddr == dev.ieeeAddr;
          });
        }
        for (output of rule.outputs) {
          updateOutputs = await _.remove(rule.outputs, function (o) {
            return o.ieeeAddr == dev.ieeeAddr;
          });
        }
        if (updateInputs.length > 0 || updateInputs.length > 0) {
          await ruleDAO.updateRule(
            { ruleId: rule.ruleId },
            { inputs: rule.inputs, outputs: rule.outputs }
          );
        }
      }
    }
  }
  if (scenarioId) {
    let rules = await ruleDAO.listRules();
    if (rules) {
      for (rule of rules) {
        if (rule.sceneOutputs.length > 0) {
          for (scenario of rule.sceneOutputs) {
            updateResult = await _.remove(rule.sceneOutputs, function (o) {
              return o.scenarioId == scenarioId;
            });
            if (updateResult.length > 0) {
              await ruleDAO.updateRule(
                { ruleId: rule.ruleId },
                { sceneOutputs: rule.sceneOutputs }
              );
            }
          }
        }
      }
    }
  }
}

