const chalk = require("chalk");
const gtwModel = require("../../../../database/models/gateway.model");
const shepherd = require("../../shepherd/zs-shepherd");
const firmwareProcessor = require("../../services/processor.firmware");
const deviceDAO = require("../../../../database/models/DAO/device.dao");
const environmentDAO = require("../../../../database/models/DAO/environment.dao");
const ruleDAO = require("../../../../database/models/DAO/rule.dao");
const scenarioDAO = require("../../../../database/models/DAO/scenario.dao");
const userAccessDAO = require("../../../../database/models/DAO/user.access.dao");
const userBlackListDAO = require("../../../../database/models/DAO/user.blackList.dao");

module.exports.execConfig = execConfig;
module.exports.execUpdate = execUpdate;
module.exports.execRefresh = execRefresh;
module.exports.execReset = execReset;
module.exports.execJoin = execJoin;
module.exports.execSync = execSync;

async function execConfig(nJSON) {
  console.log(chalk.green("[mqtt-service] config"));
  const gateway = await gtwModel.findOne({});
  if (gateway.status == "inactive") {
    await clearGateway();
  }
  gateway.name = nJSON.name;
  gateway.serialNumber = nJSON.serialNumber;
  gateway.status = "active";
  gateway.timezone = nJSON.timezone;
  gateway.timezone.offset = nJSON.timezone.offset*(-1);
  gateway.save();

  console.log(chalk.blue("[s-gateway] gateway: " + gateway._id));

  return {
    name: gateway.name,
    ipV4: gateway.ipV4,
    serialNumber: gateway.serialNumber,
    timezone: gateway.timezone,
    gatewayId: gateway._id
  };
}

async function execRefresh() {
  console.log(chalk.green("[mqtt-service] Refreshing gateway..."));

  const refreshRes = await shepherd.refresh();

  return refreshRes;
}

async function execUpdate(nJSON) {
  console.log(chalk.green("[mqtt-service] Updating gateway..."));
  const startUpdate = await firmwareProcessor.setImageStatus(3);
  const updateRes = await firmwareProcessor.updateFirmware();

  return updateRes;
}

async function execReset(nJSON) {
  console.log(chalk.green("[mqtt-service] reset gateway " + nJSON.mode));

  const resetRes = await shepherd.reset(nJSON.mode);

  if (nJSON.mode == 0) {
    const deviceRes = await deviceDAO.deleteAllDevices();
    console.log(chalk.gray("[nedb] devices deleted."));

    const environmentRes = await environmentDAO.deleteAllEnvironment();
    console.log(chalk.gray("[nedb] environments deleted."));

    const ruleRes = await ruleDAO.deleteAllRules();
    console.log(chalk.gray("[nedb] rules deleted."));

    const scenarioRes = await scenarioDAO.deleteAllScenario();
    console.log(chalk.gray("[nedb] scenario deleted."));

    const userAccessRes = await userAccessDAO.deleteAllAccess();
    console.log(chalk.gray("[nedb] user access deleted."));

    const userBlackListRes = await userBlackListDAO.deleteAllBlackList();
    console.log(chalk.gray("[nedb] user blackList deleted."));

    const gateway = await gtwModel.findOne({});
    gateway.name = "Gateway Pixel Ti";
    gateway.status = "inactive";
    gateway.save();
    console.log(chalk.gray("[nedb] gateway deactivated."));
  }
  return resetRes;
}

async function execJoin(nJSON) {
  console.log(chalk.green("[mqtt-service] permit join"));

  const result = await shepherd.join(nJSON.time);

  return result;
}

async function execSync() {
  console.log(chalk.green("[mqtt-service] list"));

  const shepList = JSON.parse(await shepherd.list());

  for (shepDevice of shepList) {
    console.log(
      chalk.blue(
        "[s-gateway] [sync-shepherd] " +
          shepDevice.type +
          ": " +
          shepDevice.ieeeAddr +
          ", " +
          shepDevice.status +
          ", " +
          shepDevice.nwkAddr +
          "."
      )
    );
  }

  const nedbList = await deviceDAO.listDevices();
  var nedbListAdjusted = [];

  for (nedbDevice of nedbList) {
    console.log(
      chalk.blue(
        "[s-gateway] [sync-nedb-b4s] " +
          nedbDevice.type +
          ": " +
          nedbDevice.ieeeAddr +
          ", " +
          nedbDevice.status +
          ", " +
          nedbDevice.nwkAddr +
          ", " +
          nedbDevice.environment +
          "."
      )
    );
    nedbListAdjusted.push(dataAdjust(nedbDevice));
  }

  return nedbListAdjusted;
}

async function clearGateway() {
  console.log(chalk.green("[mqtt-service] clearing gateway..."));

  const deviceRes = await deviceDAO.deleteAllDevices();
  console.log(chalk.gray("[nedb] devices deleted."));

  const environmentRes = await environmentDAO.deleteAllEnvironment();
  console.log(chalk.gray("[nedb] environments deleted."));

  const ruleRes = await ruleDAO.deleteAllRules();
  console.log(chalk.gray("[nedb] rules deleted."));

  const scenarioRes = await scenarioDAO.deleteAllScenario();
  console.log(chalk.gray("[nedb] scenario deleted."));

  const userAccessRes = await userAccessDAO.deleteAllAccess();
  console.log(chalk.gray("[nedb] user access deleted."));

  const userBlackListRes = await userBlackListDAO.deleteAllBlackList();
  console.log(chalk.gray("[nedb] user blackList deleted."));
}

function dataAdjust(data) {
  return {
    name: data.name.trim(),
    ieeeAddr: data.ieeeAddr,
    epList: data.epList,
    nwkAddr: data.nwkAddr,
    profId: data.profId,
    devId: data.devId,
    modelId: data.modelId.trim(),
    status: data.status,
    manufId: data.manufId,
    manufName: data.manufName,
    powerSource: data.powerSource,
    type: data.type,
    clusters: JSON.stringify(data.clusters),
    environment: data.environment,
    isActive: data.isActive,
    inClusterList: data.inClusterList,
    outClusterList: data.outClusterList
  };
}
